<?php
declare(strict_types=1);

namespace App\Controllers;

use App\Models\Claim;
use App\Models\Setting;
use App\Services\Auth;
use App\Services\CSRF;
use App\Services\DB;
use App\Services\RateLimiter;
use App\Services\Turnstile;
use App\Services\Util;

final class ClaimController
{
    public function claim(): void
    {
        CSRF::validateOrFail();
        $u = Auth::requireUser();
        if (empty($u['email_verified_at'])) {
            echo 'Verify your email before claiming.';
            return;
        }
        if (($u['status'] ?? 'active') !== 'active') {
            echo 'Account not active.';
            return;
        }

        $ip = Util::clientIp();
        $claimBucket = 'claim:' . Util::hmac($ip);
        if (!RateLimiter::hit($claimBucket, 60, 3600)) {
            http_response_code(429);
            echo 'Rate limited. Slow down.';
            return;
        }

        $tsToken = (string)($_POST['cf_turnstile_response'] ?? '');
        if (!Turnstile::verify($tsToken, $ip)) {
            echo 'Bot check failed.';
            return;
        }

        $pdo = DB::pdo();
        $pdo->beginTransaction();

        // Lock user row to prevent double-claim via rapid clicks
        $stmt = $pdo->prepare('SELECT id, next_claim_at, balance_doge FROM users WHERE id=:id FOR UPDATE');
        $stmt->execute([':id'=>(int)$u['id']]);
        $row = $stmt->fetch();
        if (!$row) { $pdo->rollBack(); echo 'User missing'; return; }

        $now = new \DateTimeImmutable('now');
        $nextClaimAt = $row['next_claim_at'] ? new \DateTimeImmutable($row['next_claim_at']) : null;
        if ($nextClaimAt && $now < $nextClaimAt) {
            $pdo->rollBack();
            echo 'Cooldown active. Try again later.';
            return;
        }

        $today = $now->format('Y-m-d');
        $countToday = Claim::countToday((int)$u['id'], $today);
        $maxPerDay = (int)Setting::get('max_claims_per_day', 5);
        if ($countToday >= $maxPerDay) {
            $pdo->rollBack();
            echo 'Daily limit reached.';
            return;
        }

        // Roll 1..100 server-side
        $rolled = random_int(1, 100);
        $reward = $this->computeReward($rolled);

        // Optional caps
        $dailyUserCap = (string)Setting::get('daily_user_cap_doge', '0');
        if ($this->gt($dailyUserCap, '0')) {
            $sumStmt = $pdo->prepare('SELECT COALESCE(SUM(reward_doge),0) AS s FROM claims WHERE user_id=:u AND claim_date=:d');
            $sumStmt->execute([':u'=>(int)$u['id'], ':d'=>$today]);
            $sum = (string)($sumStmt->fetch()['s'] ?? '0');
            if ($this->gte($this->add($sum, $reward), $dailyUserCap)) {
                $pdo->rollBack();
                echo 'Daily DOGE cap reached.';
                return;
            }
        }

        $dailySiteCap = (string)Setting::get('daily_site_cap_doge', '0');
        if ($this->gt($dailySiteCap, '0')) {
            $sumStmt = $pdo->prepare('SELECT COALESCE(SUM(reward_doge),0) AS s FROM claims WHERE claim_date=:d');
            $sumStmt->execute([':d'=>$today]);
            $sum = (string)($sumStmt->fetch()['s'] ?? '0');
            if ($this->gte($this->add($sum, $reward), $dailySiteCap)) {
                $pdo->rollBack();
                echo 'Faucet is empty for today. Try tomorrow.';
                return;
            }
        }

        $ipHash = Util::hmac($ip);
        Claim::insert((int)$u['id'], $rolled, $reward, $ipHash);
        $pdo->prepare('UPDATE users SET balance_doge = balance_doge + :a WHERE id=:id')->execute([':a'=>$reward, ':id'=>(int)$u['id']]);

        $intervalMin = (int)Setting::get('claim_interval_minutes', 15);
        $next = $now->modify('+' . $intervalMin . ' minutes')->format('Y-m-d H:i:s');
        $pdo->prepare('UPDATE users SET next_claim_at = :n WHERE id=:id')->execute([':n'=>$next, ':id'=>(int)$u['id']]);

        $pdo->commit();

        echo 'Rolled ' . $rolled . ' → +' . $reward . ' DOGE';
    }

    private function computeReward(int $roll): string
    {
        $table = Setting::get('payout_table_json', null);
        if (!is_array($table)) {
            // Default payout table
            $table = [
                ['min'=>1, 'max'=>5, 'reward'=>'0.50'],
                ['min'=>6, 'max'=>20, 'reward'=>'0.15'],
                ['min'=>21, 'max'=>60, 'reward'=>'0.05'],
                ['min'=>61, 'max'=>100, 'reward'=>'0.01'],
            ];
        }
        foreach ($table as $row) {
            $min = (int)($row['min'] ?? 0);
            $max = (int)($row['max'] ?? 0);
            if ($roll >= $min && $roll <= $max) {
                return $this->normAmount((string)($row['reward'] ?? '0'));
            }
        }
        return '0';
    }

    private function normAmount(string $a): string
    {
        // normalize to 8 dp string
        if (!preg_match('/^\d+(\.\d+)?$/', $a)) return '0';
        if (strpos($a,'.')===false) return $a . '.00000000';
        [$i,$d] = explode('.', $a, 2);
        $d = substr($d . '00000000', 0, 8);
        return $i . '.' . $d;
    }

    // Simple decimal helpers (string), good enough for small amounts
    private function add(string $a, string $b): string
    {
        if (function_exists('bcadd')) return bcadd($a, $b, 8);
        return (string)number_format(((float)$a + (float)$b), 8, '.', '');
    }
    private function gt(string $a, string $b): bool
    {
        if (function_exists('bccomp')) return bccomp($a, $b, 8) === 1;
        return (float)$a > (float)$b;
    }
    private function gte(string $a, string $b): bool
    {
        if (function_exists('bccomp')) return bccomp($a, $b, 8) >= 0;
        return (float)$a >= (float)$b;
    }
}
